﻿/****** Object:  StoredProcedure [puma].[PatientMasterGet]    Script Date: 06/20/2013 10:12:00 ******/
DROP PROCEDURE [puma].[PatientMasterGet]
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
-- =============================================
-- Create date:   23 January 2013
-- Description:   Gets one or more records from the Patient Master table,
--                applying overrides from puma.PatientMasterOverride
--                if a record is present in the latter.
-- Lookup Types Supported:
--    1. PatientMasterId (e.g. 1234567)
--    2. StatePatientId (e.g. '100000090751')
--    3. Ihi (e.g. '8003604567901177')
--    4. HospitalId and Mrn  (e.g. 71, '01234567')
--    5. HospitalCode and CodeSystemId and Mrn ('FMC', 2, '01234567')
--    6. Demographic search using all or some of:
--         CurrentLastName, CurrentFirstNames, DateOfBirth, CurrentSexId, DvaNumber, MedicareNumber and MedicareIrn
--       For demographic search
--
-- Modified: 24/03/2015	changed HealthProviderOrganisationNetworkId to return from HealthProviderOrganisation (hpo) not PatientMasterIhi (pmi)
-- =============================================
CREATE PROCEDURE [puma].[PatientMasterGet] 
(
	@PatientMasterId INT = NULL,
	@StatePatientId VARCHAR(64) = NULL,
	@Ihi VARCHAR(16) = NULL,
	@HospitalId INT = NULL,
	@HospitalCode VARCHAR(64) = NULL,
	@CodeSystemId INT = NULL,
	@Mrn VARCHAR(20) = NULL,
	@CurrentLastName VARCHAR(80) = NULL,
	@CurrentFirstNames VARCHAR(80) = NULL,
	@DateOfBirth DATETIME = NULL,
	@CurrentSexId INT = NULL,
	@MedicareNumber VARCHAR(12) = NULL,
	@MedicareIrn VARCHAR(3) = NULL,
	@DvaNumber VARCHAR(18) = NULL,
	@HealthProviderOrganisationNetworkId INT = NULL
)
AS
BEGIN
	IF @HospitalId IS NULL AND @CodeSystemId IS NOT NULL AND @HospitalCode IS NOT NULL 
	BEGIN
	  SELECT @HospitalId = [HospitalId]
	  FROM [hips].[HospitalCode]
	  WHERE [CodeSystemId] = @CodeSystemId
	  AND [Code] = @HospitalCode
	END


	IF @PatientMasterId IS NULL
	BEGIN
		IF @HospitalId IS NOT NULL AND @Mrn IS NOT NULL
		BEGIN
			SELECT	@PatientMasterId = [PatientMasterId]
			FROM	[hips].[HospitalPatient]
			WHERE	[HospitalId] = @HospitalId
			AND		[Mrn] = @Mrn
		END
		ELSE IF @Ihi IS NOT NULL
		BEGIN
			SELECT	@PatientMasterId = [PatientMasterId]
			FROM	[hips].[PatientMasterIhi]
			WHERE	[Ihi] = @Ihi
		END
		ELSE IF @StatePatientId IS NOT NULL
		BEGIN
			SELECT	@PatientMasterId = [PatientMasterId]
			FROM	[hips].[PatientMaster]
			WHERE	[StatePatientId] = @StatePatientId
		END				
	END	
	
	IF @HealthProviderOrganisationNetworkId IS NULL AND @HospitalId IS NULL AND @HospitalCode IS NULL
	BEGIN
		RAISERROR(50003, -1, -1) 
	END
	
	IF @HealthProviderOrganisationNetworkId IS NULL AND @HospitalId IS NOT NULL
	BEGIN
		SELECT @HealthProviderOrganisationNetworkId = hpo.HealthProviderOrganisationNetworkId
		FROM [hips].HealthProviderOrganisation hpo
		INNER JOIN [hips].[Hospital] h ON hpo.HealthProviderOrganisationId = h.HealthProviderOrganisationId
		WHERE h.HospitalId = @HospitalId
	END
	
	IF @HealthProviderOrganisationNetworkId IS NULL AND @HospitalId IS NULL AND @CodeSystemId IS NOT NULL AND @HospitalCode IS NOT NULL 
	BEGIN
		SELECT @HealthProviderOrganisationNetworkId = hpo.HealthProviderOrganisationNetworkId
		FROM [hips].HealthProviderOrganisation hpo
		INNER JOIN [hips].[Hospital] h ON hpo.HealthProviderOrganisationId = h.HealthProviderOrganisationId
		INNER JOIN [hips].[HospitalCode] hc ON h.HospitalId = hc.HospitalId
		WHERE hc.CodeSystemId = @CodeSystemId AND hc.Code = @HospitalCode
	END

	IF  @PatientMasterId IS NULL
	AND @Mrn IS NULL
	AND @StatePatientId IS NULL
	AND @Ihi IS NULL
	AND (  @CurrentFirstNames IS NOT NULL 
		OR @CurrentLastName IS NOT NULL
		OR @DateOfBirth IS NOT NULL
		OR @MedicareNumber IS NOT NULL
		OR @DvaNumber IS NOT NULL)
	BEGIN
		-- Demographic Search
		SELECT pm.[PatientMasterId]
			 , COALESCE (pmi.RegisteredSexId,pm.CurrentSexId, -1) AS RegisteredSexId
			 , COALESCE (pm.CurrentSexId, -1) AS CurrentSexId
			 , pmi.Ihi
			 , COALESCE (pmi.IhiStatusId, -1) AS IhiStatusId
			 , ihis.Description AS IhiStatus
			 , COALESCE (pmi.IhiRecordStatusId, -1) AS IhiRecordStatusId
			 , pmi.[RegisteredFamilyName]
			 , pmi.[RegisteredGivenName]
			 , irs.Description AS IhiRecordStatus
			 , pmi.DateLastValidated AS IhiLastValidated
			 , COALESCE (pmo.[MedicareNumber], pm.[MedicareNumber]) AS MedicareNumber
			 , COALESCE (pmo.[MedicareIrn], pm.[MedicareIrn]) AS MedicareIrn
			 , COALESCE (pmo.[IsMedicareNumberValid], pm.[IsMedicareNumberValid]) AS IsMedicareNumberValid
			 , COALESCE (pmo.[DvaNumber], pm.[DvaNumber]) AS DvaNumber
			 , pm.[StatePatientId]
			 , COALESCE (pmo.[DateOfBirth], pm.[DateOfBirth]) AS DateOfBirth
			 , COALESCE (pmi.RegisteredDateOfBirth, pm.DateOfBirth) AS RegisteredDateOfBirth
			 , hpo.[HealthProviderOrganisationNetworkId]
			 , pm.[DateOfDeath]
			 , pm.[DateCreated]
			 , pm.[UserCreated]
			 , pm.[DateModified]
			 , pm.[UserModified]
        FROM	[hips].[PatientMaster] AS pm
		LEFT JOIN [hips].HospitalPatient AS hp ON pm.PatientMasterID = hp.PatientMasterID
		LEFT JOIN [hips].[Hospital] AS h ON h.HospitalID = hp.HospitalID
		LEFT JOIN hips.HealthProviderOrganisation AS hpo
			ON h.HealthProviderOrganisationId = hpo.HealthProviderOrganisationId	
        LEFT JOIN [hips].[PatientMasterIhi] AS pmi ON pmi.[PatientMasterID] = pm.[PatientMasterID]
			AND pmi.HealthProviderOrganisationNetworkId = @HealthProviderOrganisationNetworkId
		LEFT JOIN [hips].PatientMasterName pmn ON pmn.PatientMasterId = pm.PatientMasterId AND pmn.NameTypeId = 2
		LEFT JOIN [hips].IhiRecordStatus AS irs on irs.IhiRecordStatusId = pmi.IhiRecordStatusId
		LEFT JOIN [hips].IhiStatus AS ihis on ihis.IhiStatusId = pmi.IhiStatusId
		LEFT JOIN [puma].PatientMasterOverride AS pmo ON pmo.PatientMasterId = pm.PatientMasterId
		
        WHERE	(@CurrentLastName IS NULL OR pmn.[FamilyName] = @CurrentLastName)
		AND		(@CurrentFirstNames IS NULL OR pmn.[GivenNames] = @CurrentFirstNames)
		AND		(@DateOfBirth IS NULL OR pm.[DateOfBirth] = @DateOfBirth)
		AND		(@CurrentSexId IS NULL OR pm.[CurrentSexId] = @CurrentSexId)
		AND		(@DvaNumber IS NULL OR pm.[DvaNumber] = @DvaNumber)
		-- Do not match a patient master where the query and the record both have one of the special values that mean unknown or ineligible.
		AND		(@MedicareNumber IS NULL OR pm.[MedicareNumber] = @MedicareNumber AND @MedicareNumber <> '0000000000' AND @MedicareNumber <> '0000000009')

		-- Do match even if the record has a null sequence number
		AND		(@MedicareIrn IS NULL OR pm.[MedicareIrn] IS NULL OR pm.[MedicareIrn] = @MedicareIrn)
	END
	ELSE  -- @PatientMasterID IS NOT NULL
	BEGIN
		SELECT pm.[PatientMasterID]
			 , COALESCE (pmi.RegisteredSexId,pm.CurrentSexId, -1) AS RegisteredSexId
			 , sex.Description as RegisteredSex
			 , COALESCE (pm.CurrentSexId, -1) AS CurrentSexId
			 , pmi.Ihi
			 , COALESCE (pmi.IhiStatusId, -1) AS IhiStatusId
			 , ihis.Description AS IhiStatus
			 , COALESCE (pmi.IhiRecordStatusId, -1) AS IhiRecordStatusId
			 , irs.Description AS IhiRecordStatus
			 , pmi.[RegisteredFamilyName]
			 , pmi.[RegisteredGivenName]
			 , pmi.DateLastValidated AS IhiLastValidated
			 , COALESCE (pmo.[MedicareNumber], pm.[MedicareNumber]) AS MedicareNumber
			 , COALESCE (pmo.[MedicareIrn], pm.[MedicareIrn]) AS MedicareIrn
			 , COALESCE (pmo.[IsMedicareNumberValid], pm.[IsMedicareNumberValid]) AS IsMedicareNumberValid
			 , COALESCE (pmo.[DvaNumber], pm.[DvaNumber]) AS DvaNumber
			 , pm.[StatePatientId]
			 , COALESCE (pmo.[DateOfBirth], pm.[DateOfBirth]) AS DateOfBirth
			 , COALESCE (pmi.RegisteredDateOfBirth, pm.DateOfBirth) AS RegisteredDateOfBirth
			 , hpo.[HealthProviderOrganisationNetworkId]
			 , pm.[DateOfDeath]
			 , pm.[DateCreated]
			 , pm.[UserCreated]
			 , pm.[DateModified]
			 , pm.[UserModified]
		FROM	[hips].[PatientMaster] AS pm
		LEFT JOIN [hips].HospitalPatient AS p ON pm.PatientMasterID = p.PatientMasterID
		LEFT JOIN [hips].[Hospital] AS h ON h.HospitalID = p.HospitalID
		LEFT JOIN hips.HealthProviderOrganisation AS hpo
			ON h.HealthProviderOrganisationId = hpo.HealthProviderOrganisationId	
        LEFT JOIN [hips].[PatientMasterIhi] AS pmi ON pmi.[PatientMasterID] = pm.[PatientMasterID]
			AND pmi.HealthProviderOrganisationNetworkId = @HealthProviderOrganisationNetworkId
		LEFT JOIN [hips].IhiRecordStatus AS irs on irs.IhiRecordStatusId = pmi.IhiRecordStatusId
		LEFT JOIN [hips].IhiStatus AS ihis on ihis.IhiStatusId = pmi.IhiStatusId
		LEFT JOIN [puma].PatientMasterOverride AS pmo ON pmo.PatientMasterId = pm.PatientMasterId
		LEFT JOIN [hips].Sex as sex ON sex.SexId = COALESCE (pmi.RegisteredSexId,pm.CurrentSexId, -1)
        WHERE	pm.[PatientMasterID] = @PatientMasterID
		AND     (@DateOfBirth IS NULL OR @DateOfBirth = pm.[DateOfBirth])
	END
END

GO
